% Code for: Capital Reallocation and Aggregate Productivity
% R. Cooper & I. Schott
% Code version: September 2022
function [res, PARMS, steady]  = steady_state_k(PARMS)

% Define data moments
define_data_moms

%% Use the current parameter guess to build grids &c.

% Idiosyncratic shocks --> Poisson process
mu              = -PARMS.sigma/2;
eps             = mu + linspace(-3*PARMS.sigma,3*PARMS.sigma,PARMS.N)';
% Distribution:
PARMS.PP        = normpdf(eps,mu,PARMS.sigma) / sum(normpdf(eps,mu,PARMS.sigma));
PARMS.eps       = exp(eps);
PARMS.meaneps   = PARMS.PP'*PARMS.eps;
% The implied transition matrix (Poisson)
prob_rho        = zeros(PARMS.N);
prob_rho(1:PARMS.N+1:end)   = PARMS.rho_eps;
prob_rho        = prob_rho + PARMS.PP'.*(1-PARMS.rho_eps);
PARMS.prob_rho  = prob_rho;
clear prob_rho eps mu

%% Frictionless benchmark

% Compute the first-best solution (frictionless case)
mm              = PARMS.PP' * (PARMS.eps.^(1/(1-PARMS.ALPHA))); % stuff under the integral
kmax            = PARMS.eps.^(1/(1-PARMS.ALPHA)) ./ mm(:)'; % Frictionless capital vector

% % Compute frictionless values
Y_fl            = (PARMS.PP'*(PARMS.eps.*(kmax.^PARMS.ALPHA)))'; % net of capital
mu_fl           = (PARMS.PP'*(kmax.^PARMS.ALPHA))';
phi_fl          = Y_fl - PARMS.meaneps.*mu_fl;
x               = (1-PARMS.B)/PARMS.B + PARMS.DELTA;
kstar_fl        = (Y_fl*PARMS.ALPHA./x).^(1/(1-PARMS.ALPHA));

%% Solution of steady state

% For a given choice of pi, the solution is to maximize present discounted
% output net of the adjustment costs, subject to the feasibility constraint
% that capital must sum up to one:

% Guess initial pi and run loop until k and pi are jointly optimal
% Update steady state solution
err     = 1;
err2    = 1;
err3    = 1; % capital

pistar  = datamoms{1,1}; %#ok<USENS> % guess = target
mu_a    = mu_fl;
phi_a   = phi_fl;
A       = 1; % aggregate productivity
kstar   = kstar_fl;

while abs(err) > 1e-10 || abs(err2) > 1e-10 || abs(err3) > 1e-10

    cstar   = A*kstar^PARMS.ALPHA*(PARMS.meaneps*mu_a + phi_a*pistar/(1 - PARMS.rho_eps*(1-pistar))) + (1-PARMS.DELTA)*kstar - kstar - kstar*pistar.^2/2*PARMS.F_upper;
    % Build infinite sum term of parameters
    zeta_i  = pistar*PARMS.ALPHA/cstar*A*kstar^PARMS.ALPHA*((PARMS.eps - PARMS.meaneps)/(1-PARMS.B*PARMS.rho_eps*(1-pistar)) + PARMS.meaneps/(1-PARMS.B*(1-pistar)) );

    % Lagrange multiplier on equality constraint
    lambda  = (PARMS.PP'* (zeta_i.^(1/(1-PARMS.ALPHA))) )^(1-PARMS.ALPHA);
    % Optimal k-vector given the guess for pi
    k_sol   = (zeta_i / lambda ).^(1/(1-PARMS.ALPHA));

    % Update moments
    mu_a    = PARMS.PP'*k_sol.^PARMS.ALPHA;
    tmp     = PARMS.PP'*(PARMS.eps.*k_sol.^PARMS.ALPHA) - PARMS.meaneps*mu_a;
    err2    = phi_a - tmp;
    phi_a   = tmp; %

    % Given k, find the optimal pi
    % Write steady state consumption as a function of pi
    c       = @(x) A*kstar^PARMS.ALPHA*(PARMS.meaneps*mu_a + phi_a*x/(1 - PARMS.rho_eps*(1-x))) + (1-PARMS.DELTA)*kstar - kstar - kstar*x.^2/2*PARMS.F_upper;

    % The numerical FOC for pi
    foc_pi  = @(x) 1/c(x) * (- kstar*x*PARMS.F_upper + A*kstar^PARMS.ALPHA*phi_a*((1-PARMS.rho_eps)/(1 - PARMS.rho_eps*(1-x))) /(1 - PARMS.B*PARMS.rho_eps*(1-x)) );

    % Error using the previous solution
    err     = foc_pi(pistar);
    % Updated guess
    pistar  = fzero(foc_pi,pistar);

    % Update the optimal kstar
    x       = (1-PARMS.B)/PARMS.B + PARMS.DELTA + pistar^2/2 * PARMS.F_upper;
    SR      = A*(PARMS.meaneps*mu_a + phi_a*pistar/(1 - PARMS.rho_eps*(1-pistar)));
    new     = (SR*PARMS.ALPHA./x).^(1/(1-PARMS.ALPHA));
    err3    = kstar - new;
    kstar   = new;

end

%% Compute distribution & moments

compute_stationary_distribution

% Compute stationary equilibrium
compute_ss_moments

%% Solution
% Steady state moments
y_a     = PARMS.PP'*(PARMS.eps.*k_sol.^PARMS.ALPHA); % output by adjusters
mu_a    = PARMS.PP' * k_sol.^PARMS.ALPHA;
phi_a   = y_a - PARMS.meaneps*PARMS.PP'*k_sol.^PARMS.ALPHA; %

mu_n    = mu_a;
phi_n   = phi_a.*pistar./(pistar + (1-PARMS.rho_eps)/PARMS.rho_eps);

% Consumption
cstar           = A*kstar^PARMS.ALPHA*((PARMS.meaneps*mu_a + phi_a)*pistar + (PARMS.meaneps*mu_n + phi_n)*(1-pistar)) + (1-PARMS.DELTA)*kstar - kstar - kstar*pistar^2/2 * PARMS.F_upper;
steady.xss      = [mu_n; phi_n; 0; kstar]; % mu, phi, A innovation, capital
steady.yss      = [k_sol; pistar; cstar]; % k-vector, pi, consumption

% Save the steady state values
steady.pistar   = pistar;
steady.k_a      = k_sol;
steady.y_a      = y_a;
steady.cstar    = cstar;
steady.mu_a     = mu_a;
steady.phi_a    = phi_a;
steady.mu_n     = mu_n;
steady.phi_n    = phi_n;
steady.probst   = probst;
steady.trans_pi = trans_pi;
steady.trans_nopi = trans_nopi;
steady.R        = R_ss;
steady.sig      = stdlogYK;
steady.kstar    = kstar;

%% Collect model moments
% Moments for the stationary distribution
simmoms{1,1} = pistar;   % Fraction of firms adjusting capital
simmoms{2,1} = R_ss;   % Average capital reallocation
simmoms{3,1} = stdlogtfp; % std log TFPR, output weighted

% Untargeted
simmoms{4,1} = rho_lS; % AR(1) of log sales
simmoms{5,1} = corr_pi_eps(2);  % Correlation between productivity and reallocation rate
simmoms{6,1} = rho_lR; % Serial correlation of R
simmoms{7,1} = R_fractions(1); % Importance of small capital reallocation rates

tmp_data    = cat(1,datamoms{:,1});
tmp_model   = cat(1,simmoms{:,1});

% Stationary model
selection   = [1 2 3]; %
res         = mean(abs((tmp_data(selection)-tmp_model(selection))./tmp_data(selection)));
fprintf('\n The distance between data and targeted model moments is %4.3f%%\n', 100*res  )




